const express = require('express');
const screenshot = require('screenshot-desktop');
const Jimp = require('jimp');
const { exec, execSync } = require("child_process");
const fs = require('fs');
const bcrypt = require('bcryptjs');
const path = require('path');
const cron = require('node-cron');
const os = require('os');
const https = require('https');

// Funktion zum Auflösen von Pfaden (kompatibel mit pkg)
function resolvePath(filePath, isExternal = false) {
  if (process.pkg && !isExternal) {
    return path.join(__dirname, filePath);
  }
  return path.resolve(process.pkg ? path.dirname(process.execPath) : __dirname, filePath);
}

// Funktion zum Lesen von Dateien
function readFileSync(filePath, isExternal = false) {
  const resolvedPath = resolvePath(filePath, isExternal);
  return fs.readFileSync(resolvedPath, 'utf8');
}

// Funktion zum Schreiben von Dateien
function writeFileSync(filePath, data, isExternal = true) {
  const resolvedPath = resolvePath(filePath, isExternal);
  fs.writeFileSync(resolvedPath, data, 'utf8');
}

// Lade CSS (intern)
const picoCSS = readFileSync('pico.min.css');

// Pfad zu nircmd.exe (extern)
const nircmdPath = resolvePath('nircmd.exe', true);

const PORT = process.env.PORT || 54168;
const HOST = process.env.HOST || '0.0.0.0';
const SCREEN_WIDTH = process.env.SCREEN_WIDTH || 1000;

const app = express();
let displayTimes = { enabled: false, schedule: [] };
let passwordHash = null;
let onTimeTasks = [];
let offTimeTasks = [];
let browserUrl = '';
let restartOnNoInternet = false;
let internetCheckInterval = null;
let debugMode = false;


// Initialize
loadSettings();


function setScreenTimes(rules) {
  if (!rules.enabled) {
    unSetScreenTimes();
    console.log('Screen times disabled');
    return;
  }

  var weekdays = '';
  if(rules.schedule && rules.schedule[0] && rules.schedule[0].days.length) {
    var _weekdays = [];
    rules.schedule[0].days.forEach(day => {
      switch (day) {
        case "Montag": _weekdays.push(1); break;
        case "Dienstag": _weekdays.push(2); break;
        case "Mittwoch": _weekdays.push(3); break;
        case "Donnerstag": _weekdays.push(4); break;
        case "Freitag": _weekdays.push(5); break;
        case "Samstag": _weekdays.push(6); break;
        case "Sonntag": _weekdays.push(0); break;
      }
    });
    weekdays = _weekdays.join(',');
  } else {
    weekdays = '*'
  }
  
  var onTime, offTime;
  if(rules.schedule && rules.schedule[0]) {
    let from = rules.schedule[0].startTime.split(':');
    let to = rules.schedule[0].endTime.split(':');
    onTime = `${Number(from[1])} ${Number(from[0])}`;
    offTime = `${Number(to[1])} ${Number(to[0])}`;
  } else {
    onTime = '00 00';
    offTime = '59 23';
  }
      
  unSetScreenTimes();
          
  var onCommand = process.platform === "win32" ? `${nircmdPath} sendmouse move 1 1` : 'vcgencmd display_power 1';
  console.log(`turn on at: ${onTime} * * ${weekdays}`);
  onTimeTasks.push(
    cron.schedule(`${onTime} * * ${weekdays}`, () => {
      exec(onCommand);
      console.log('turn display on');
    })
  );
  
  var offCommand = process.platform === "win32" ? `${nircmdPath} monitor async_off` : 'vcgencmd display_power 0';
  console.log(`turn off at: ${offTime} * * ${weekdays}`);
  offTimeTasks.push(
    cron.schedule(`${offTime} * * ${weekdays}`, () => {
      exec(offCommand);
      console.log('turn display off');
    })
  );

  console.log('Screen times set:', rules);
}

function unSetScreenTimes() {
  onTimeTasks.forEach(task => task.stop());
  offTimeTasks.forEach(task => task.stop());
  onTimeTasks = [];
  offTimeTasks = [];
}

// Funktion zum Überprüfen, ob ein Programm installiert ist
function isInstalled(program) {
  try {
    if (process.platform === "win32") {
      execSync(`where ${program}`);
    } else {
      execSync(`which ${program}`);
    }
    return true;
  } catch (e) {
    return false;
  }
}

// Funktion zur Überprüfung der Internetverbindung
function checkInternetConnection() {
  return new Promise((resolve) => {
    const testUrl = 'https://detectportal.aushang.net/success.txt';

    const req = https.get(testUrl, (res) => {
      if (res.statusCode !== 200) {
        console.log(`Unexpected status code: ${res.statusCode}`);
        resolve(false);
        return;
      }

      let data = '';
      res.on('data', (chunk) => {
        data += chunk;
      });

      res.on('end', () => {
        if (data.trim() === 'success') {
          console.log("Internet connection is available");
          resolve(true);
        } else {
          console.log("Unexpected response content");
          resolve(false);
        }
      });
    });

    req.on('error', (error) => {
      console.log(`Error checking internet connection: ${error.message}`);
      resolve(false);
    });

    req.setTimeout(5000, () => {
      req.abort();
      console.log("Connection timeout");
      resolve(false);
    });
  });
}

// Funktion zum Starten der Internetverbindungsprüfung
function startInternetCheck() {
  if (internetCheckInterval) {
    clearInterval(internetCheckInterval);
  }
  
  internetCheckInterval = setInterval(async () => {
    const isConnected = await checkInternetConnection();
    if (!isConnected && restartOnNoInternet) {
      if (debugMode) {
        console.log('DEBUG: Would restart due to no internet connection');
      } else {
        console.log('No internet connection detected. Restarting...');
        reboot();
      }
    }
  }, 120000); // Überprüfung alle 2 Minuten (120000 ms)
}

// Funktion zum Stoppen der Internetverbindungsprüfung
function stopInternetCheck() {
  if (internetCheckInterval) {
    clearInterval(internetCheckInterval);
    internetCheckInterval = null;
  }
}

// Starte die Internetverbindungsprüfung, wenn die Option aktiviert ist
if (restartOnNoInternet) {
  startInternetCheck();
}

// Funktion zum Neustarten des Systems
function reboot() {
  if (debugMode) {
    console.log('DEBUG: Would reboot the system');
    return;
  }

  const command = process.platform === "win32" ? 
    'shutdown /r /t 10 /c "aushang.net wird aufgrund fehlender Internetverbindung neu gestartet"' :
    'shutdown -r +1 "aushang.net wird aufgrund fehlender Internetverbindung neu gestartet"';
  
  exec(command, (error, stdout, stderr) => {
    if (error || stderr) {
      console.error(`Reboot error: ${error || stderr}`);
    } else {
      console.log(`Reboot scheduled in 1 minute due to no internet connection`);
    }
  });
}


// Funktion zum Schließen des Browsers
async function closeBrowser() {
  return new Promise((resolve, reject) => {
    let command;
    if (process.platform === "win32") {
      command = 'taskkill /F /IM msedge.exe';
    } else if (process.platform === "linux") {
      if (isInstalled('firefox')) {
        command = 'pkill firefox';
      } else if (isInstalled('google-chrome')) {
        command = 'pkill chrome';
      } else if (isInstalled('chromium-browser')) {
        command = 'pkill chromium';
      }
    }

    if (!command) {
      console.log('No browser close command available for this platform');
      resolve();
      return;
    }

    exec(command, (error, stdout, stderr) => {
      if (error && error.code !== 128) { // Ignore error code 128 (no process found)
        console.error(`Error closing browser: ${error}`);
        reject(error);
      } else {
        console.log('Browser closed successfully');
        resolve();
      }
    });
  });
}

// Funktion zum Öffnen einer URL im Browser
async function openUrlInBrowser(url) {
  if (!url) return Promise.resolve();

  let command;
  if (process.platform === "win32") {
    command = `powershell -Command "Start-Process microsoft-edge:${url} -ArgumentList '--autoplay-policy=no-user-gesture-required'; Start-Sleep -Seconds 2; Add-Type -AssemblyName System.Windows.Forms; [System.Windows.Forms.SendKeys]::SendWait('{F11}');"`;
  } else if (process.platform === "linux") {
    if (isInstalled('firefox')) {
      command = `firefox --new-window ${url}`; // set in `about:config` attribute `media.autoplay.default` to `0`
    } else if (isInstalled('google-chrome')) {
      command = `google-chrome --start-fullscreen ${url} --autoplay-policy=no-user-gesture-required`;
    } else if (isInstalled('chromium-browser')) {
      command = `chromium-browser --start-fullscreen ${url} --autoplay-policy=no-user-gesture-required`;
    } else {
      return Promise.reject(new Error('No supported browser found'));
    }
  } else {
    return Promise.reject(new Error('Unsupported platform'));
  }

  return new Promise((resolve, reject) => {
    exec(command, (error, stdout, stderr) => {
      if (error) {
        console.error(`Error opening URL: ${error.message}`);
        reject(error);
      } else {
        console.log(`URL opened in browser: ${url}`);
        browserUrl = url;
        storeSettings();
        resolve(true);
      }
    });
  });
}

// Funktion zum Einschalten des Displays
async function turnDisplayOn() {
  if (debugMode) {
    console.log('DEBUG: Screen on command received');
    return Promise.resolve(true);
  }

  let commands = [];
  if (process.platform === "win32") {
    commands.push(`${nircmdPath} sendmouse move 1 1`);
  } else if (process.platform === "linux") {
    commands = [
      'wlr-randr --output HDMI-A-1 --on',
      'wlr-randr --output HDMI-A-2 --on',
      'vcgencmd display_power 1'
    ];
  } else if (process.platform === "darwin") {
    commands.push('caffeinate -u -t 2');
  }

  return new Promise((resolve, reject) => {
    const tryCommands = async (commands) => {
      if (commands.length === 0) {
        console.log('Kein Befehl konnte erfolgreich ausgeführt werden.');
        reject(new Error('No command succeeded'));
        return;
      }

      const command = commands.shift();
      try {
        await execAsync(command);
        console.log(`Display turned on using command: ${command}`);

        // Nach dem Einschalten Browser neu starten wenn URL vorhanden
        const data = readFileSync('config.json', true);
        const settings = JSON.parse(data);
        console.log(settings);
        console.log(settings.browserUrl);
        if (settings.browserUrl) {
          await closeBrowser();
          await openUrlInBrowser(settings.browserUrl);
        }
        resolve(true);
      } catch (error) {
        console.log(`Error with command "${command}": ${error.message}`);
        tryCommands(commands);
      }
    };

    tryCommands([...commands]); // Copy array to avoid mutation
  });
}

// Funktion zum Ausschalten des Displays
async function turnDisplayOff() {
  if (debugMode) {
    console.log('DEBUG: Screen off command received');
    return Promise.resolve(true);
  }

  // Erst Browser schließen
  await closeBrowser();

  let commands = [];
  if (process.platform === "win32") {
    commands.push(`${nircmdPath} monitor async_off`);
  } else if (process.platform === "linux") {
    commands = [
      'vcgencmd display_power 0',
      'xset dpms force off',
      'echo 1 | sudo tee /sys/class/backlight/*/bl_power',
      'wlr-randr --output HDMI-A-1 --off',
      'wlr-randr --output HDMI-A-2 --off',
    ];
  } else if (process.platform === "darwin") {
    commands.push('pmset displaysleepnow');
  }

  return new Promise((resolve, reject) => {
    const tryCommands = async (commands) => {
      if (commands.length === 0) {
        console.log('Kein Befehl konnte erfolgreich ausgeführt werden.');
        reject(new Error('No command succeeded'));
        return;
      }

      const command = commands.shift();
      try {
        await execAsync(command);
        console.log(`Display turned off using command: ${command}`);
        resolve(true);
      } catch (error) {
        console.log(`Error with command "${command}": ${error.message}`);
        tryCommands(commands);
      }
    };

    tryCommands([...commands]); // Copy array to avoid mutation
  });
}

// Helper function to promisify exec
function execAsync(command) {
  return new Promise((resolve, reject) => {
    exec(command, (error, stdout, stderr) => {
      if (error) {
        reject(error);
      } else if (stderr) {
        reject(new Error(stderr));
      } else {
        resolve(stdout);
      }
    });
  });
}

// Funktion zum Laden der Einstellungen
function loadSettings() {
  try {
    const data = readFileSync('config.json', true);
    const settings = JSON.parse(data);
    if (settings.displayTimes) {
      displayTimes = settings.displayTimes;
      setScreenTimes(displayTimes);
    }
    if (settings.passwordHash) {
      passwordHash = settings.passwordHash;
      console.log('Loaded existing password hash');
    }
    if (settings.browserUrl) {
      browserUrl = settings.browserUrl;
      console.log('Loaded browser URL:', browserUrl);
      openUrlInBrowser(browserUrl);
    }
    if (settings.restartOnNoInternet) {
      restartOnNoInternet = settings.restartOnNoInternet;
    }
    console.log('Settings loaded successfully');
  } catch (err) {
    console.error('Error loading settings:', err);
  }
 
  if (!passwordHash) {
    console.log('No password set. Please set a password on first use.');
  } else {
    console.log('Existing password hash found');
  }
}

// Funktion zum Speichern der Einstellungen
function storeSettings() {
  const settings = {
    displayTimes,
    passwordHash,
    browserUrl,
    restartOnNoInternet,
  };
  try {
    writeFileSync('config.json', JSON.stringify(settings, null, 2), true);
    console.log('Settings saved successfully');
  } catch (err) {
    console.error('Error saving settings:', err);
  }
}

// Funktion zum Setzen der Display-Zeiten
function setScreenTimes(rules) {
  if (!rules.enabled) {
    unSetScreenTimes();
    console.log('Screen times disabled');
    return;
  }

  var weekdays = '';
  if(rules.schedule && rules.schedule[0] && rules.schedule[0].days.length) {
    var _weekdays = [];
    rules.schedule[0].days.forEach(day => {
      switch (day) {
        case "Montag": _weekdays.push(1); break;
        case "Dienstag": _weekdays.push(2); break;
        case "Mittwoch": _weekdays.push(3); break;
        case "Donnerstag": _weekdays.push(4); break;
        case "Freitag": _weekdays.push(5); break;
        case "Samstag": _weekdays.push(6); break;
        case "Sonntag": _weekdays.push(0); break;
      }
    });
    weekdays = _weekdays.join(',');
  } else {
    weekdays = '*'
  }
  
  var onTime, offTime;
  if(rules.schedule && rules.schedule[0]) {
    let from = rules.schedule[0].startTime.split(':');
    let to = rules.schedule[0].endTime.split(':');
    onTime = `${Number(from[1])} ${Number(from[0])}`;
    offTime = `${Number(to[1])} ${Number(to[0])}`;
  } else {
    onTime = '00 00';
    offTime = '59 23';
  }
      
  unSetScreenTimes();
          
  console.log(`turn on at: ${onTime} * * ${weekdays}`);
  onTimeTasks.push(
    cron.schedule(`${onTime} * * ${weekdays}`, async () => {
      await turnDisplayOn();
    })
  );
  
  console.log(`turn off at: ${offTime} * * ${weekdays}`);
  offTimeTasks.push(
    cron.schedule(`${offTime} * * ${weekdays}`, async () => {
      await turnDisplayOff();
    })
  );

  console.log('Screen times set:', rules);
}

function unSetScreenTimes() {
  onTimeTasks.forEach(task => task.stop());
  offTimeTasks.forEach(task => task.stop());
  onTimeTasks = [];
  offTimeTasks = [];
}

// Express Middleware für Passwortschutz
app.use(express.json());
app.use((req, res, next) => {
  if (req.path === '/' || req.path === '/login' || req.path === '/set-password') return next();
  const token = req.headers['authorization'];
  if (passwordHash && token && bcrypt.compareSync(token, passwordHash)) {
    next();
  } else {
    res.status(401).send('Unauthorized');
  }
});

// Weboberfläche
app.get('/', (req, res) => {
  function getLocalIpAddress() {
    const interfaces = os.networkInterfaces();
    for (const devName in interfaces) {
      const iface = interfaces[devName];
      for (let i = 0; i < iface.length; i++) {
        const alias = iface[i];
        if (alias.family === 'IPv4' && alias.address !== '127.0.0.1' && !alias.internal) {
          return alias.address;
        }
      }
    }
    return '0.0.0.0'; // Fallback, falls keine IP gefunden wurde
  }

  const localIpAddress = getLocalIpAddress();
  const port = PORT;

  let html = `
    <html>
      <head>
        <title>aushang.net</title>
        <style>${picoCSS}</style>
        <style>
          .text-center {text-align: center;}
          .mb-4	{margin-bottom: 1rem;}
          h3 {margin-top: 2rem;}
          .subtitle {
            font-size: 1.2rem;
            color: #666;
            margin-top: -1rem;
            margin-bottom: 2rem;
          }
          #notification {
            position: fixed;
            top: 20px;
            right: 20px;
            padding: 10px 20px;
            background-color: #4CAF50;
            color: white;
            border-radius: 5px;
            display: none;
          }
        </style>
      </head>
      <body>
        <main class="container">
          <h1 class="text-center">http://${localIpAddress}:${port}</h1>
          <p class="text-center subtitle">aushang.net</p>
          <div id="passwordForm">
            <input type="password" id="newPassword" placeholder="Passwort setzen" required>
            <button onclick="setPassword()">Passwort setzen</button>
          </div>
          <form id="loginForm">
            <input type="password" id="password" placeholder="Passwort" required>
            <button type="submit">Anmelden</button>
          </form>
          <div id="controls">
            <div class="text-center mb-4">
              <button onclick="takeScreenshot()">Screenshot</button>
              <button onclick="shutdown()">Herunterfahren</button>
              <button onclick="reboot()">Neustart</button>
              <button onclick="screenOn()">Bildschirm einschalten</button>
              <button onclick="screenOff()">Bildschirm ausschalten</button>
            </div>
            <h3>Neustart bei fehlender Internetverbindung</h3>
            <div>
              <label>
                <input type="checkbox" id="restartOnNoInternet" onchange="toggleRestartOnNoInternet()">
                Neustart bei fehlender Internetverbindung aktivieren
              </label>
            </div>
            <h3>Display-Zeiten</h3>
            <div>
              <label>
                <input type="checkbox" id="displayTimesEnabled" onchange="toggleDisplayTimes()">
                Display-Zeiten aktivieren
              </label>
              <div id="displayTimesSettings" style="display:none;">
                <div id="weekdays" class="grid"></div>
                <div class="grid">
                  <input type="time" step="60" id="startTime">
                  <input type="time" step="60" id="endTime">
                </div>
                <button onclick="setDisplayTimes()">Speichern</button>
              </div>
            </div>
            <h3>URL im Browser öffnen</h3>
            <div>
              <input type="url" id="browserUrl" placeholder="https://wp.aushang.net/demo?secret=geheim">
              <button onclick="openUrlInBrowser()">Öffnen</button>
            </div>
          </div>
          <div id="notification"></div>
        </main>
        <script>
          let token = localStorage.getItem('token');
          const weekdays = ['Montag', 'Dienstag', 'Mittwoch', 'Donnerstag', 'Freitag', 'Samstag', 'Sonntag'];
          let displayTimesEnabled = ${displayTimes.enabled};
          let browserUrl = '${browserUrl || ''}';
          let restartOnNoInternet = ${restartOnNoInternet};
          
          function updateUI() {
            if (token) {
              document.getElementById('controls').style.display = 'block';
              document.getElementById('loginForm').style.display = 'none';
              document.getElementById('passwordForm').style.display = 'none';
              document.getElementById('displayTimesEnabled').checked = displayTimesEnabled;
              document.getElementById('displayTimesSettings').style.display = displayTimesEnabled ? 'block' : 'none';
              document.getElementById('restartOnNoInternet').checked = restartOnNoInternet;
            } else if ('${passwordHash}' === 'null') {
              document.getElementById('passwordForm').style.display = 'block';
              document.getElementById('loginForm').style.display = 'none';
              document.getElementById('controls').style.display = 'none';
            } else {
              document.getElementById('loginForm').style.display = 'block';
              document.getElementById('passwordForm').style.display = 'none';
              document.getElementById('controls').style.display = 'none';
            }
            
            if (browserUrl) {
              document.getElementById('browserUrl').value = browserUrl;
            }
          }

          function showNotification(message) {
            const notification = document.getElementById('notification');
            notification.textContent = message;
            notification.style.display = 'block';
            setTimeout(() => {
              notification.style.display = 'none';
            }, 3000);
          }

          function setPassword() {
            const password = document.getElementById('newPassword').value;
            fetch('/set-password', {
              method: 'POST',
              headers: { 'Content-Type': 'application/json' },
              body: JSON.stringify({ password })
            }).then(response => {
              if (response.ok) {
                showNotification('Passwort gesetzt');
                location.reload();
              }
            });
          }

          function login(e) {
            e.preventDefault();
            const password = document.getElementById('password').value;
            fetch('/login', {
              method: 'POST',
              headers: { 'Content-Type': 'application/json' },
              body: JSON.stringify({ password })
            }).then(response => response.json())
            .then(data => {
              if (data.token) {
                token = data.token;
                localStorage.setItem('token', token);
                updateUI();
              }
            });
          }

          function takeScreenshot() {
            fetch('/screen.png', {
              headers: { 'Authorization': token }
            }).then(response => response.blob())
            .then(blob => {
              const url = window.URL.createObjectURL(blob);
              window.open(url, '_blank');
            });
          }

          function shutdown() {
            if (confirm('Wirklich herunterfahren?')) {
              fetch('/shutdown', {
                headers: { 'Authorization': token }
              });
            }
          }

          function reboot() {
            if (confirm('Wirklich neustarten?')) {
              fetch('/reboot', {
                headers: { 'Authorization': token }
              });
            }
          }

          function screenOn() {
            fetch('/screen-on', {
              headers: { 'Authorization': token }
            }).then(response => response.json())
            .then(data => {
              if (data.success) {
                showNotification('Bildschirm eingeschaltet');
              } else {
                showNotification('Fehler beim Einschalten des Bildschirms');
              }
            });
          }

          function screenOff() {
            fetch('/screen-off', {
              headers: { 'Authorization': token }
            }).then(response => response.json())
            .then(data => {
              if (data.success) {
                showNotification('Bildschirm ausgeschaltet');
              } else {
                showNotification('Fehler beim Ausschalten des Bildschirms');
              }
            });
          }

          function toggleDisplayTimes() {
            displayTimesEnabled = document.getElementById('displayTimesEnabled').checked;
            document.getElementById('displayTimesSettings').style.display = displayTimesEnabled ? 'block' : 'none';
            setDisplayTimes();
          }

          function setDisplayTimes() {
            const selectedDays = Array.from(document.querySelectorAll('#weekdays input:checked')).map(cb => cb.value);
            const startTime = document.getElementById('startTime').value;
            const endTime = document.getElementById('endTime').value;
            
            fetch('/set-display-times', {
              method: 'POST',
              headers: { 
                'Authorization': token,
                'Content-Type': 'application/json'
              },
              body: JSON.stringify({ 
                enabled: displayTimesEnabled,
                schedule: displayTimesEnabled ? [{
                  days: selectedDays,
                  startTime: startTime,
                  endTime: endTime
                }] : []
              })
            }).then(() => showNotification('Display-Zeiten aktualisiert'));
          }

          function openUrlInBrowser() {
            const url = document.getElementById('browserUrl').value;
            if (!url) {
              showNotification('Bitte geben Sie eine gültige URL ein.');
              return;
            }
            
            fetch('/open-url', {
              method: 'POST',
              headers: { 
                'Authorization': token,
                'Content-Type': 'application/json'
              },
              body: JSON.stringify({ url })
            })
            .then(response => response.json())
            .then(data => {
              if (data.success) {
                showNotification('URL im Browser geöffnet');
                browserUrl = url;
              } else {
                showNotification('Fehler beim Öffnen der URL: ' + data.error);
              }
            });
          }

          function toggleRestartOnNoInternet() {
            restartOnNoInternet = document.getElementById('restartOnNoInternet').checked;
            fetch('/set-restart-on-no-internet', {
              method: 'POST',
              headers: { 
                'Authorization': token,
                'Content-Type': 'application/json'
              },
              body: JSON.stringify({ restartOnNoInternet })
            })
            .then(response => response.json())
            .then(data => {
              if (data.success) {
                showNotification(restartOnNoInternet ? 'Neustart bei fehlender Internetverbindung aktiviert' : 'Neustart bei fehlender Internetverbindung deaktiviert');
              } else {
                showNotification('Fehler beim Ändern der Neustart-Einstellung');
              }
            });
          }
          
          document.getElementById('loginForm').addEventListener('submit', login);

          // Erzeuge Checkboxen für Wochentage
          const weekdaysDiv = document.getElementById('weekdays');
          weekdays.forEach(day => {
            const label = document.createElement('label');
            const checkbox = document.createElement('input');
            checkbox.type = 'checkbox';
            checkbox.id = day;
            checkbox.value = day;
            label.appendChild(checkbox);
            label.appendChild(document.createTextNode(day));
            weekdaysDiv.appendChild(label);
          });

          updateUI();
        </script>
      </body>
    </html>
  `;
  res.send(html);
});

// Login-Endpoint
app.post('/login', (req, res) => {
  console.log('Login attempt');
  if (passwordHash && req.body.password && bcrypt.compareSync(req.body.password, passwordHash)) {
    console.log('Login successful');
    res.json({ token: req.body.password });
  } else {
    console.log('Login failed');
    res.status(401).send('Invalid credentials');
  }
});

// Passwort setzen
app.post('/set-password', (req, res) => {
  console.log('Attempting to set new password');
  if (!passwordHash) {
    passwordHash = bcrypt.hashSync(req.body.password, 10);
    storeSettings();
    console.log('New password set successfully');
    res.status(200).send('Password set successfully');
  } else {
    console.log('Attempt to set password failed - password already set');
    res.status(403).send('Password already set');
  }
});

app.get('/screen.png', (req, res) => {
  const width = parseInt(req.query.width) || SCREEN_WIDTH;
  screenshot({ format: 'png' })
    .then((img) => {
      console.log('Take Screenshot');
      Jimp.read(img)
        .then((image) => {
          return image.resize(width, Jimp.AUTO).getBufferAsync(Jimp.MIME_PNG);
        })
        .then((smallImg) => {
          res.writeHead(200, { 'Content-type': 'image/png' });
          res.end(smallImg);
        })
        .catch((err) => {
          console.warn('Image processing error:', err);
          res.status(500).end();
        });
    })
    .catch((err) => {
      console.warn('Screenshot error:', err);
      res.status(500).end();
    });
});

app.get('/shutdown', (req, res) => {
  if (debugMode) {
    console.log('DEBUG: Shutdown command received');
    res.json({ success: true, debug: true }).end();
    return;
  }
  const command = process.platform === "win32" ? 
    'shutdown /s /t 60 /c "aushang.net wird ausgeschaltet"' :
    'shutdown -h 1';
  
  exec(command, (error, stdout, stderr) => {
    if (error || stderr) {
      console.error(`Shutdown error: ${error || stderr}`);
      res.json({ success: false }).end();
    } else {
      console.log(`Shutdown scheduled in 1 minute`);
      res.json({ success: true }).end();
    }
  });
});

app.get('/reboot', (req, res) => {
  if (debugMode) {
    console.log('DEBUG: Reboot command received');
    res.json({ success: true, debug: true }).end();
    return;
  }
  const command = process.platform === "win32" ? 
    'shutdown /r /t 60 /c "aushang.net wird neu gestartet"' :
    'shutdown -r 1';
  
  exec(command, (error, stdout, stderr) => {
    if (error || stderr) {
      console.error(`Reboot error: ${error || stderr}`);
      res.json({ success: false }).end();
    } else {
      console.log(`Reboot scheduled in 1 minute`);
      res.json({ success: true }).end();
    }
  });
});

app.get('/screen-off', async (req, res) => {
  try {
    await turnDisplayOff();
    res.json({ success: true });
  } catch (error) {
    res.json({ success: false, error: error.message });
  }
});

app.get('/screen-on', async (req, res) => {
  try {
    await turnDisplayOn();
    res.json({ success: true });
  } catch (error) {
    res.json({ success: false, error: error.message });
  }
});

app.post('/open-url', async (req, res) => {
  const { url } = req.body;
  if (!url) {
    return res.status(400).json({ success: false, message: 'URL is required' });
  }

  try {
    await openUrlInBrowser(url);
    res.json({ success: true, message: 'URL opened in browser' });
  } catch (error) {
    res.status(500).json({ success: false, message: 'Failed to open URL', error: error.message });
  }
});

// Endpunkt für Displayzeiten
app.post('/set-display-times', (req, res) => {
  console.log('Updating display times:', req.body);
  displayTimes = req.body;
  setScreenTimes(displayTimes);
  storeSettings();
  res.send('Display times updated');
});

// Endpunkt für Neustart bei fehlender Internetverbindung
app.post('/set-restart-on-no-internet', (req, res) => {
  restartOnNoInternet = req.body.restartOnNoInternet;
  storeSettings();
  if (restartOnNoInternet) {
    startInternetCheck();
  } else {
    stopInternetCheck();
  }
  res.json({ success: true });
});

// Starte den Server
app.listen(PORT, HOST, () => {
  console.log(`Server running on http://${HOST}:${PORT}`);
  console.log(`Password status: ${passwordHash ? 'Set' : 'Not set'}`);
});

